<?php
/**
 * ═══════════════════════════════════════════════════
 * 📦 Order Tracking Module
 * ═══════════════════════════════════════════════════
 */

require_once HELPERS_DIR . 'database.php';
require_once HELPERS_DIR . 'copywriting.php';

/**
 * نمایش سفارشات کاربر
 */
function showUserOrders($chat_id, $user_id) {
    $orders = getUserOrders($user_id);
    
    if(empty($orders)) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "
📦 <b>سفارشات شما</b>

شما هنوز سفارشی ثبت نکرده‌اید.

💡 از منوی «خدمات» سرویس موردنظرتان را انتخاب کنید!
",
            'parse_mode' => 'HTML'
        ]);
        return;
    }
    
    $keyboard = [];
    $message = "📦 <b>سفارشات شما</b>\n\n";
    
    foreach($orders as $order_id => $order) {
        $status_emoji = [
            'pending' => '⏳',
            'processing' => '🔄',
            'completed' => '✅',
            'cancelled' => '❌'
        ];
        
        $emoji = $status_emoji[$order['status']] ?? '❓';
        
        $message .= "$emoji <b>$order_id</b>\n";
        $message .= "📋 {$order['service']}\n";
        $message .= "💰 " . number_format($order['amount']) . " تومان\n";
        $message .= "📅 " . date('Y/m/d H:i', $order['created_at']) . "\n";
        $message .= "━━━━━━━━━━━━\n\n";
        
        $keyboard[] = [
            ['text' => "🔍 جزئیات $order_id", 'callback_data' => "order_details_$order_id"]
        ];
    }
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode([
            'inline_keyboard' => $keyboard
        ])
    ]);
}

/**
 * نمایش جزئیات سفارش
 */
function showOrderDetails($chat_id, $order_id) {
    $order = getOrder($order_id);
    
    if(!$order) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "❌ سفارش یافت نشد."
        ]);
        return;
    }
    
    $status_text = getOrderStatusMessage($order['status']);
    
    $message = "
🔍 <b>جزئیات سفارش</b>

━━━━━━━━━━━━━━━━━━━━
🆔 <b>شناسه:</b> <code>$order_id</code>
📦 <b>سرویس:</b> {$order['service']}
💰 <b>مبلغ:</b> " . number_format($order['amount']) . " تومان
💳 <b>روش پرداخت:</b> " . ($order['payment_method'] == 'online' ? 'آنلاین' : 'کارت به کارت') . "

━━━━━━━━━━━━━━━━━━━━
📊 <b>وضعیت:</b>
$status_text

━━━━━━━━━━━━━━━━━━━━
📅 <b>تاریخ ثبت:</b> " . date('Y/m/d H:i', $order['created_at']) . "
🔄 <b>آخرین بروزرسانی:</b> " . date('Y/m/d H:i', $order['updated_at']) . "
";
    
    if(!empty($order['notes'])) {
        $message .= "\n━━━━━━━━━━━━━━━━━━━━\n📝 <b>یادداشت‌ها:</b>\n\n";
        foreach($order['notes'] as $note) {
            $message .= "• " . $note['text'] . "\n";
            $message .= "  ⏰ " . date('Y/m/d H:i', $note['time']) . "\n\n";
        }
    }
    
    $keyboard = [
        [['text' => '💬 پشتیبانی', 'callback_data' => 'contact_support']],
        [['text' => '🔙 بازگشت به سفارشات', 'callback_data' => 'my_orders']]
    ];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * اطلاع‌رسانی به کاربر در هنگام تغییر وضعیت
 */
function notifyOrderStatusChange($order_id, $new_status, $note = '') {
    $order = getOrder($order_id);
    if(!$order) return;
    
    $user_id = $order['user_id'];
    $status_message = getOrderStatusMessage($new_status);
    
    $message = "
🔔 <b>بروزرسانی سفارش</b>

🆔 <b>شناسه سفارش:</b> <code>$order_id</code>

$status_message
";
    
    if($note) {
        $message .= "\n💬 <b>توضیحات:</b>\n$note";
    }
    
    apiRequest('sendMessage', [
        'chat_id' => $user_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode([
            'inline_keyboard' => [
                [['text' => '🔍 مشاهده جزئیات', 'callback_data' => "order_details_$order_id"]]
            ]
        ])
    ]);
}

/**
 * پنل مدیریت سفارشات برای ادمین
 */
function showAdminOrderPanel($chat_id) {
    $orders = loadOrders();
    
    $stats = [
        'total' => count($orders),
        'pending' => 0,
        'processing' => 0,
        'completed' => 0,
        'cancelled' => 0
    ];
    
    foreach($orders as $order) {
        $stats[$order['status']]++;
    }
    
    $message = "
👨‍💼 <b>پنل مدیریت سفارشات</b>

━━━━━━━━━━━━━━━━━━━━
📊 <b>آمار کلی:</b>

📦 کل سفارشات: {$stats['total']}
⏳ در انتظار: {$stats['pending']}
🔄 در حال انجام: {$stats['processing']}
✅ تکمیل شده: {$stats['completed']}
❌ لغو شده: {$stats['cancelled']}

━━━━━━━━━━━━━━━━━━━━
";
    
    $keyboard = [
        [
            ['text' => '⏳ سفارشات در انتظار', 'callback_data' => 'admin_orders_pending'],
            ['text' => '🔄 در حال انجام', 'callback_data' => 'admin_orders_processing']
        ],
        [
            ['text' => '✅ تکمیل شده', 'callback_data' => 'admin_orders_completed'],
            ['text' => '❌ لغو شده', 'callback_data' => 'admin_orders_cancelled']
        ],
        [
            ['text' => '🔍 جستجو سفارش', 'callback_data' => 'admin_search_order']
        ]
    ];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * نمایش سفارشات بر اساس وضعیت (برای ادمین)
 */
function showOrdersByStatus($chat_id, $status) {
    $orders = loadOrders();
    $filtered = array_filter($orders, function($order) use ($status) {
        return $order['status'] == $status;
    });
    
    if(empty($filtered)) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "هیچ سفارشی با وضعیت «$status» یافت نشد."
        ]);
        return;
    }
    
    $message = "📋 <b>سفارشات با وضعیت: $status</b>\n\n";
    $keyboard = [];
    
    foreach($filtered as $order_id => $order) {
        $message .= "🆔 <code>$order_id</code>\n";
        $message .= "👤 کاربر: {$order['user_id']}\n";
        $message .= "📦 {$order['service']}\n";
        $message .= "💰 " . number_format($order['amount']) . " تومان\n";
        $message .= "📅 " . date('Y/m/d H:i', $order['created_at']) . "\n";
        $message .= "━━━━━━━━━━━━\n\n";
        
        $keyboard[] = [
            ['text' => "مدیریت $order_id", 'callback_data' => "admin_manage_$order_id"]
        ];
    }
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}
?>