<?php
/**
 * ═══════════════════════════════════════════════════
 * 💳 Payment Gateway Module (ZarinPal) - FIXED
 * ═══════════════════════════════════════════════════
 */

require_once HELPERS_DIR . 'database.php';
require_once HELPERS_DIR . 'copywriting.php';

/**
 * ایجاد درخواست پرداخت در زرین‌پال
 */
function createZarinpalPayment($order_id, $amount, $description, $user_id) {
    // ✅ FIX: metadata رو حذف کردیم
    $data = [
        'merchant_id' => ZARINPAL_MERCHANT,
        'amount' => $amount,
        'description' => $description,
        'callback_url' => ZARINPAL_CALLBACK . "?order_id=$order_id&user_id=$user_id"
    ];
    
    logActivity("Creating ZarinPal payment - Order: $order_id, Amount: $amount");
    
    $ch = curl_init('https://api.zarinpal.com/pg/v4/payment/request.json');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $result = curl_exec($ch);
    $error = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if($error) {
        logActivity("ZarinPal cURL Error: $error");
        return false;
    }
    
    $result = json_decode($result, true);
    
    if(isset($result['data']['code']) && $result['data']['code'] == 100) {
        $authority = $result['data']['authority'];
        
        // ذخیره اطلاعات پرداخت
        $payment_id = createPayment($order_id, $amount, 'online');
        updatePayment($payment_id, [
            'authority' => $authority,
            'status' => 'pending'
        ]);
        
        logActivity("Payment created successfully - Order: $order_id, Authority: $authority");
        
        return 'https://www.zarinpal.com/pg/StartPay/' . $authority;
    }
    
    // خطا در پاسخ زرین‌پال
    $error_json = json_encode($result);
    logActivity("ZarinPal Error: $error_json");
    
    return false;
}

/**
 * تایید پرداخت در زرین‌پال
 */
function verifyZarinpalPayment($authority, $amount) {
    $data = [
        'merchant_id' => ZARINPAL_MERCHANT,
        'authority' => $authority,
        'amount' => $amount
    ];
    
    logActivity("Verifying payment - Authority: $authority, Amount: $amount");
    
    $ch = curl_init('https://api.zarinpal.com/pg/v4/payment/verify.json');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $result = curl_exec($ch);
    curl_close($ch);
    
    logActivity("Verification Response: $result");
    
    $result = json_decode($result, true);
    
    if(isset($result['data']['code']) && ($result['data']['code'] == 100 || $result['data']['code'] == 101)) {
        return [
            'success' => true,
            'ref_id' => $result['data']['ref_id'],
            'card_pan' => $result['data']['card_pan'] ?? ''
        ];
    }
    
    return ['success' => false, 'message' => $result['errors']['message'] ?? 'خطای نامشخص'];
}

/**
 * نمایش گزینه‌های پرداخت
 */
function showPaymentOptions($chat_id, $order_id) {
    $order = getOrder($order_id);
    
    if(!$order) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "❌ سفارش یافت نشد."
        ]);
        return;
    }
    
    $message = getPaymentMethodsMessage();
    
    $keyboard = [
        [['text' => '💳 پرداخت آنلاین (زرین‌پال)', 'callback_data' => "pay_online_$order_id"]],
        [['text' => '💵 کارت به کارت', 'callback_data' => "pay_card_$order_id"]],
        [['text' => '🔙 بازگشت', 'callback_data' => 'my_orders']]
    ];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * پرداخت آنلاین
 */
function processOnlinePayment($chat_id, $user_id, $order_id) {
    $order = getOrder($order_id);
    
    if(!$order) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "❌ سفارش یافت نشد."
        ]);
        return;
    }
    
    if($order['status'] != 'pending') {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "⚠️ این سفارش قبلاً پرداخت شده است."
        ]);
        return;
    }
    
    $amount = $order['amount'];
    $description = "پرداخت سفارش $order_id - {$order['service']}";
    
    logActivity("Processing online payment - User: $user_id, Order: $order_id");
    
    $payment_url = createZarinpalPayment($order_id, $amount, $description, $user_id);
    
    if(!$payment_url) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "
❌ <b>خطا در ایجاد درگاه پرداخت</b>

لطفاً چند لحظه دیگر مجدداً تلاش کنید یا از روش کارت به کارت استفاده نمایید.

💬 در صورت ادامه مشکل با پشتیبانی تماس بگیرید.

🔧 <b>کد خطا:</b> <code>$order_id</code>
📝 این کد را برای پشتیبانی ارسال کنید.
",
            'parse_mode' => 'HTML',
            'reply_markup' => json_encode([
                'inline_keyboard' => [
                    [['text' => '💵 کارت به کارت', 'callback_data' => "pay_card_$order_id"]],
                    [['text' => '💬 پشتیبانی', 'callback_data' => 'contact_support']]
                ]
            ])
        ]);
        return;
    }
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "
✅ <b>درگاه پرداخت آماده شد!</b>

━━━━━━━━━━━━━━━━━━━━
🆔 <b>شناسه سفارش:</b> <code>$order_id</code>
💰 <b>مبلغ قابل پرداخت:</b> " . number_format($amount) . " تومان

━━━━━━━━━━━━━━━━━━━━
🔐 <b>پرداخت از طریق درگاه امن زرین‌پال</b>

✔️ پشتیبانی از تمام کارت‌های بانکی
✔️ امنیت بالا و رمزنگاری SSL
✔️ تایید خودکار پرداخت

⚠️ <b>مهم:</b> پس از پرداخت، به همین ربات بازگردید تا سفارش شما فعال شود.

👇 برای پرداخت روی دکمه زیر کلیک کنید:
",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode([
            'inline_keyboard' => [
                [['text' => '💳 پرداخت ' . number_format($amount) . ' تومان', 'url' => $payment_url]],
                [['text' => '❌ انصراف', 'callback_data' => 'my_orders']]
            ]
        ])
    ]);
    
    logActivity("Payment URL sent to user $user_id for order $order_id - URL: $payment_url");
}

/**
 * پرداخت کارت به کارت
 */
function processCardToCardPayment($chat_id, $user_id, $order_id) {
    $order = getOrder($order_id);
    
    if(!$order) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "❌ سفارش یافت نشد."
        ]);
        return;
    }
    
    $amount = $order['amount'];
    $message = getCardToCardMessage();
    
    // ذخیره حالت انتظار رسید
    file_put_contents(DATA_DIR . "wait_receipt_$user_id.txt", $order_id);
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message . "\n\n💰 <b>مبلغ قابل پرداخت:</b> " . number_format($amount) . " تومان",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode([
            'keyboard' => [
                [['text' => '🔙 بازگشت']]
            ],
            'resize_keyboard' => true
        ])
    ]);
}

/**
 * پردازش رسید پرداخت
 */
function processReceiptUpload($chat_id, $user_id, $message) {
    $wait_file = DATA_DIR . "wait_receipt_$user_id.txt";
    
    if(!file_exists($wait_file)) {
        return false;
    }
    
    $order_id = trim(file_get_contents($wait_file));
    unlink($wait_file);
    
    $order = getOrder($order_id);
    if(!$order) return false;
    
    // ذخیره رسید
    $receipt_data = $user_id . '|' . time() . '|' . $order_id . PHP_EOL;
    file_put_contents(RECEIPTS_FILE, $receipt_data, FILE_APPEND);
    
    // ارسال رسید به ادمین
    $caption = "
📥 <b>رسید پرداخت جدید</b>

━━━━━━━━━━━━━━━━━━━━
👤 <b>کاربر:</b> <a href='tg://user?id=$user_id'>$user_id</a>
🆔 <b>سفارش:</b> <code>$order_id</code>
📦 <b>سرویس:</b> {$order['service']}
💰 <b>مبلغ:</b> " . number_format($order['amount']) . " تومان
🕐 <b>تاریخ:</b> " . date('Y/m/d H:i') . "
";
    
    $admin_keyboard = [
        [
            ['text' => '✅ تایید', 'callback_data' => "approve_payment_$order_id"],
            ['text' => '❌ رد', 'callback_data' => "reject_payment_$order_id"]
        ],
        [
            ['text' => '💬 پاسخ به کاربر', 'callback_data' => "reply_user_$user_id"]
        ]
    ];
    
    // ارسال رسید به ادمین
    if(isset($message['photo'])) {
        $photo = end($message['photo']);
        apiRequest('sendPhoto', [
            'chat_id' => ADMIN_ID,
            'photo' => $photo['file_id'],
            'caption' => $caption,
            'parse_mode' => 'HTML',
            'reply_markup' => json_encode(['inline_keyboard' => $admin_keyboard])
        ]);
    } elseif(isset($message['document'])) {
        apiRequest('sendDocument', [
            'chat_id' => ADMIN_ID,
            'document' => $message['document']['file_id'],
            'caption' => $caption,
            'parse_mode' => 'HTML',
            'reply_markup' => json_encode(['inline_keyboard' => $admin_keyboard])
        ]);
    } else {
        apiRequest('sendMessage', [
            'chat_id' => ADMIN_ID,
            'text' => $caption . "\n\n📝 متن: " . ($message['text'] ?? ''),
            'parse_mode' => 'HTML',
            'reply_markup' => json_encode(['inline_keyboard' => $admin_keyboard])
        ]);
    }
    
    // اطلاع به کاربر
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "
✅ <b>رسید شما دریافت شد!</b>

━━━━━━━━━━━━━━━━━━━━
🆔 <b>سفارش:</b> <code>$order_id</code>

رسید پرداخت شما برای تیم پشتیبانی ارسال شد و در اسرع وقت بررسی می‌شود.

⏰ <b>زمان تایید:</b> معمولاً کمتر از ۱۵ دقیقه

پس از تایید، سفارش شما فوراً شروع می‌شود.

🙏 از صبر و شکیبایی شما سپاسگزاریم!
",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode([
            'keyboard' => [
                [['text' => '📦 سفارشات من'], ['text' => '💬 پشتیبانی']],
                [['text' => '🏠 منوی اصلی']]
            ],
            'resize_keyboard' => true
        ])
    ]);
    
    logActivity("Receipt uploaded by user $user_id for order $order_id");
    
    return true;
}

/**
 * تایید پرداخت توسط ادمین
 */
function approvePayment($order_id, $admin_id) {
    $order = getOrder($order_id);
    if(!$order) return false;
    
    updateOrderStatus($order_id, 'processing', 'پرداخت تایید شد. کار شروع شده است.');
    
    $user_id = $order['user_id'];
    
    apiRequest('sendMessage', [
        'chat_id' => $user_id,
        'text' => "
🎉 <b>پرداخت شما تایید شد!</b>

━━━━━━━━━━━━━━━━━━━━
🆔 <b>سفارش:</b> <code>$order_id</code>
✅ <b>وضعیت:</b> در حال انجام

━━━━━━━━━━━━━━━━━━━━
🔄 تیم ما هم‌اکنون روی سفارش شما کار می‌کند.

⏰ <b>زمان تحویل:</b> حداکثر ۴۸ ساعت

📊 می‌توانید از منوی «سفارشات من» پیشرفت کار را ببینید.

💬 در صورت نیاز به راهنمایی، پشتیبانی در خدمت شماست.
",
        'parse_mode' => 'HTML'
    ]);
    
    apiRequest('sendMessage', [
        'chat_id' => $admin_id,
        'text' => "✅ پرداخت سفارش <code>$order_id</code> تایید شد و به کاربر اطلاع داده شد.",
        'parse_mode' => 'HTML'
    ]);
    
    logActivity("Payment approved for order $order_id by admin $admin_id");
    
    return true;
}

/**
 * رد پرداخت توسط ادمین
 */
function rejectPayment($order_id, $admin_id) {
    $order = getOrder($order_id);
    if(!$order) return false;
    
    $user_id = $order['user_id'];
    
    apiRequest('sendMessage', [
        'chat_id' => $user_id,
        'text' => "
⚠️ <b>رسید پرداخت شما تایید نشد</b>

━━━━━━━━━━━━━━━━━━━━
🆔 <b>سفارش:</b> <code>$order_id</code>

ممکن است دلایل زیر وجود داشته باشد:
• رسید خوانا نیست
• مبلغ واریزی با مبلغ سفارش مطابقت ندارد
• اطلاعات ناقص است

━━━━━━━━━━━━━━━━━━━━
💬 لطفاً با پشتیبانی تماس بگیرید یا رسید واضح‌تری ارسال کنید.
",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode([
            'inline_keyboard' => [
                [['text' => '📤 ارسال مجدد رسید', 'callback_data' => "pay_card_$order_id"]],
                [['text' => '💬 تماس با پشتیبانی', 'callback_data' => 'contact_support']]
            ]
        ])
    ]);
    
    apiRequest('sendMessage', [
        'chat_id' => $admin_id,
        'text' => "❌ پرداخت سفارش <code>$order_id</code> رد شد و به کاربر اطلاع داده شد.",
        'parse_mode' => 'HTML'
    ]);
    
    logActivity("Payment rejected for order $order_id by admin $admin_id");
    
    return true;
}
?>