<?php
/**
 * Enhanced SpotPlayer API Integration
 * Captain Trade Bot - Professional License Management
 * Updated: 2025-10-15 16:33:51 UTC
 * Created by: saportinsta65
 */

class SpotPlayerAPI {
    private $api_url;
    private $api_key;
    private $level;
    private $course_id;
    private $retry_attempts = 3;
    
    public function __construct() {
        $this->api_url = SPOTPLAYER_API_URL;
        $this->api_key = SPOTPLAYER_API_KEY;
        $this->level = SPOTPLAYER_LEVEL;
        $this->course_id = COURSE_ID;
    }
    
    /**
     * Create new license for student with enhanced options
     */
    public function createLicense($name, $mobile, $test = false, $custom_options = []) {
        $default_data = [
            'test' => $test,
            'course' => [$this->course_id],
            'name' => $this->sanitizeName($name),
            'offline' => 30, // 30 days offline access
            'payload' => $this->generatePayload($mobile),
            'watermark' => [
                'position' => 511, // All positions
                'reposition' => 15, // Change position every 15 seconds
                'margin' => 40,
                'texts' => [
                    [
                        'text' => $mobile,
                        'repeat' => 10,
                        'font' => 1,
                        'weight' => 1,
                        'color' => 2164260863, // Semi-transparent white
                        'size' => 50,
                        'stroke' => [
                            'color' => 2164260863,
                            'size' => 1
                        ]
                    ],
                    [
                        'text' => 'کاپیتان ترید',
                        'repeat' => 5,
                        'font' => 1,
                        'weight' => 2,
                        'color' => 2164260863,
                        'size' => 30,
                        'stroke' => [
                            'color' => 2164260863,
                            'size' => 1
                        ]
                    ]
                ]
            ],
            'device' => [
                'p0' => 1, // All Devices: 1 concurrent
                'p1' => 1, // Windows: 1 device
                'p2' => 1, // MacOS: 1 device  
                'p3' => 0, // Ubuntu: disabled
                'p4' => 1, // Android: 1 device
                'p5' => 1, // iOS: 1 device
                'p6' => 0  // WebApp: disabled
            ],
            'data' => [
                'confs' => 0,
                'limit' => [
                    $this->course_id => '0-' // Full access
                ]
            ]
        ];
        
        // Merge with custom options
        $license_data = array_merge($default_data, $custom_options);
        
        // Attempt to create license with retries
        for ($attempt = 1; $attempt <= $this->retry_attempts; $attempt++) {
            try {
                $response = $this->makeRequest($license_data);
                
                if ($response && isset($response['key'])) {
                    $result = [
                        'success' => true,
                        'license_id' => $response['_id'],
                        'license_key' => $response['key'],
                        'download_url' => $this->buildDownloadUrl($response['url']),
                        'course_ids' => $license_data['course'],
                        'is_test' => $license_data['test'],
                        'created_at' => date('Y-m-d H:i:s'),
                        'watermark_mobile' => $mobile
                    ];
                    
                    // Log successful creation
                    error_log("SpotPlayer license created successfully: " . $response['_id']);
                    return $result;
                }
                
                // Handle API errors
                if (isset($response['ex'])) {
                    $error_msg = $response['ex']['msg'] ?? 'Unknown API error';
                    error_log("SpotPlayer API error (attempt $attempt): $error_msg");
                    
                    if ($attempt < $this->retry_attempts) {
                        sleep(2); // Wait before retry
                        continue;
                    }
                    
                    return [
                        'success' => false,
                        'error' => $error_msg,
                        'error_code' => 'API_ERROR'
                    ];
                }
                
            } catch (Exception $e) {
                error_log("SpotPlayer request failed (attempt $attempt): " . $e->getMessage());
                
                if ($attempt < $this->retry_attempts) {
                    sleep(2); // Wait before retry
                    continue;
                }
                
                return [
                    'success' => false,
                    'error' => 'خطا در برقراری ارتباط با سرور',
                    'error_code' => 'CONNECTION_ERROR'
                ];
            }
        }
        
        return [
            'success' => false,
            'error' => 'تمام تلاش‌ها ناموفق. لطفا دوباره تلاش کنید.',
            'error_code' => 'MAX_RETRIES_EXCEEDED'
        ];
    }
    
    /**
     * Verify existing license (enhanced validation)
     */
    public function verifyLicense($license_key) {
        // Basic format validation
        if (!$this->isValidLicenseFormat($license_key)) {
            return false;
        }
        
        // Check if license exists in database (to prevent duplicates)
        global $db;
        
        try {
            $stmt = $db->connection->prepare("
                SELECT COUNT(*) as count 
                FROM users 
                WHERE license_key = ? AND is_verified = 1
            ");
            $stmt->execute([$license_key]);
            $result = $stmt->fetch();
            
            // License should be unique (not already used)
            if ($result['count'] > 0) {
                error_log("License already in use: " . substr($license_key, 0, 20) . "...");
                return false;
            }
            
            // Additional validation: Check license pattern
            return $this->validateLicensePattern($license_key);
            
        } catch (Exception $e) {
            error_log("License verification error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Validate license format
     */
    private function isValidLicenseFormat($license_key) {
        // SpotPlayer licenses are typically 128+ characters, hexadecimal
        return strlen($license_key) >= 100 && 
               strlen($license_key) <= 200 && 
               ctype_alnum($license_key);
    }
    
    /**
     * Advanced license pattern validation
     */
    private function validateLicensePattern($license_key) {
        // SpotPlayer license pattern validation
        // This is a simplified check - in production, you might want more sophisticated validation
        
        if (strlen($license_key) < 100) {
            return false;
        }
        
        // Check if it starts with typical SpotPlayer prefix
        $valid_prefixes = ['0001', '0002', '0003'];
        $prefix = substr($license_key, 0, 4);
        
        return in_array($prefix, $valid_prefixes);
    }
    
    /**
     * Make HTTP request to SpotPlayer API with enhanced error handling
     */
    private function makeRequest($data) {
        $headers = [
            'Content-Type: application/json',
            '$API: ' . $this->api_key,
            '$LEVEL: ' . $this->level,
            'User-Agent: CapitanTrade-Bot/1.0',
            'Accept: application/json'
        ];
        
        $json_data = json_encode($data);
        
        // Log request (without sensitive data)
        $log_data = $data;
        unset($log_data['watermark']); // Remove watermark from logs
        error_log("SpotPlayer API Request: " . json_encode($log_data));
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->api_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $json_data,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_FOLLOWLOCATION => false,
            CURLOPT_MAXREDIRS => 0
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        if ($curl_error) {
            error_log("SpotPlayer cURL Error: $curl_error");
            throw new Exception("Network error: $curl_error");
        }
        
        if ($http_code !== 200) {
            error_log("SpotPlayer HTTP Error: $http_code - $response");
            throw new Exception("HTTP error: $http_code");
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("SpotPlayer JSON Error: " . json_last_error_msg());
            throw new Exception("Invalid JSON response");
        }
        
        // Log response (without sensitive data)
        if (isset($result['key'])) {
            error_log("SpotPlayer license created: " . $result['_id']);
        } elseif (isset($result['ex'])) {
            error_log("SpotPlayer API Error: " . $result['ex']['msg']);
        }
        
        return $result;
    }
    
    /**
     * Build download URL
     */
    private function buildDownloadUrl($url_path) {
        if (!$url_path) {
            return 'https://dl.spotplayer.ir/';
        }
        
        return 'https://dl.spotplayer.ir' . $url_path;
    }
    
    /**
     * Sanitize name for SpotPlayer
     */
    private function sanitizeName($name) {
        // Remove emojis and special characters
        $name = preg_replace('/[\x{1F600}-\x{1F64F}]/u', '', $name);
        $name = preg_replace('/[\x{1F300}-\x{1F5FF}]/u', '', $name);
        $name = preg_replace('/[\x{1F680}-\x{1F6FF}]/u', '', $name);
        $name = preg_replace('/[\x{2600}-\x{26FF}]/u', '', $name);
        $name = preg_replace('/[\x{2700}-\x{27BF}]/u', '', $name);
        
        // Keep only letters, numbers, and basic punctuation
        $name = preg_replace('/[^\p{L}\p{N}\s\-_.]/u', '', $name);
        
        // Trim and limit length
        $name = trim($name);
        if (strlen($name) > 50) {
            $name = substr($name, 0, 50);
        }
        
        return $name ?: 'Student';
    }
    
    /**
     * Generate payload for tracking
     */
    private function generatePayload($mobile) {
        return base64_encode(json_encode([
            'mobile' => $mobile,
            'course' => 'captain_trade_advanced',
            'timestamp' => time(),
            'source' => 'telegram_bot'
        ]));
    }
    
    /**
     * Create test license (for development/testing)
     */
    public function createTestLicense($name, $mobile) {
        return $this->createLicense($name, $mobile, true, [
            'offline' => 7, // 7 days only for test
            'device' => [
                'p0' => 1, // Only one device total
                'p1' => 1, // Windows only
                'p2' => 0, 'p3' => 0, 'p4' => 0, 'p5' => 0, 'p6' => 0
            ]
        ]);
    }
    
    /**
     * Create premium license with extended features
     */
    public function createPremiumLicense($name, $mobile) {
        return $this->createLicense($name, $mobile, false, [
            'offline' => 90, // 90 days offline access
            'device' => [
                'p0' => 3, // 3 concurrent devices
                'p1' => 2, // 2 Windows devices
                'p2' => 1, // 1 macOS device
                'p3' => 0, // Ubuntu disabled
                'p4' => 2, // 2 Android devices
                'p5' => 1, // 1 iOS device
                'p6' => 0  // WebApp disabled
            ]
        ]);
    }
    
    /**
     * Get license usage statistics (if API supports it)
     */
    public function getLicenseStats($license_id) {
        // This would require a different API endpoint
        // For now, return basic info
        return [
            'license_id' => $license_id,
            'status' => 'active',
            'last_used' => date('Y-m-d H:i:s')
        ];
    }
    
    /**
     * Validate API key and connection
     */
    public function testConnection() {
        try {
            // Try to create a minimal test license
            $test_data = [
                'test' => true,
                'course' => [$this->course_id],
                'name' => 'connection_test',
                'watermark' => [
                    'texts' => [['text' => '09000000000']]
                ]
            ];
            
            $response = $this->makeRequest($test_data);
            
            if (isset($response['key'])) {
                return [
                    'success' => true,
                    'message' => 'اتصال به SpotPlayer موفقیت‌آمیز'
                ];
            }
            
            return [
                'success' => false,
                'message' => 'خطا در اتصال: ' . ($response['ex']['msg'] ?? 'Unknown error')
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'خطا در اتصال: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get API rate limits info
     */
    public function getRateLimits() {
        // This would need to be implemented based on SpotPlayer's rate limiting
        return [
            'requests_per_minute' => 60,
            'requests_per_hour' => 1000,
            'current_usage' => 0 // Would need to track this
        ];
    }
    
    /**
     * Format license for display
     */
    public function formatLicenseForDisplay($license_key) {
        if (strlen($license_key) <= 50) {
            return $license_key;
        }
        
        // Show first 20 and last 20 characters with ... in between
        return substr($license_key, 0, 20) . '...' . substr($license_key, -20);
    }
    
    /**
     * Check if license is expired (basic check)
     */
    public function isLicenseExpired($license_key, $created_date, $offline_days = 30) {
        $created_timestamp = strtotime($created_date);
        $expiry_timestamp = $created_timestamp + ($offline_days * 24 * 60 * 60);
        
        return time() > $expiry_timestamp;
    }
}
?>