<?php
/**
 * Ultimate Message Router - Brain of the Bot
 * Updated: 2025-10-15 19:17:50 UTC
 * Created by: saportinsta65
 */

class MessageRouter {
    private $telegram;
    private $db;
    private $adminCommands;
    private $fileHandler;
    private $callbackHandler;
    private $paymentProcessor;
    
    public function __construct($telegram, $db) {
        $this->telegram = $telegram;
        $this->db = $db;
        
        // Initialize handlers
        $this->adminCommands = new AdminCommands($telegram, $db);
        $this->fileHandler = new FileHandler($telegram, $db);
        $this->callbackHandler = new CallbackHandler($telegram, $db);
        $this->paymentProcessor = new PaymentProcessor($telegram, $db, new SpotPlayerAPI());
    }
    
    /**
     * 🎯 مسیریابی اصلی پیام‌ها
     */
    public function route($chat_id, $user_id, $username, $first_name, $message_text, $message_type = 'text') {
        try {
            // Show typing
            $this->telegram->sendTyping($chat_id);
            
            // Get user state and info
            $user_state = $this->db->getUserState($user_id);
            $user_info = $this->db->getUserInfo($user_id);
            
            // Log routing info
            error_log("ROUTING: user_id=$user_id, state=$user_state, text=$message_text, type=$message_type");
            
            // Handle different message types
            switch ($message_type) {
                case 'photo':
                    return $this->handlePhotoMessage($chat_id, $user_id, $message_text);
                case 'document':
                    return $this->handleDocumentMessage($chat_id, $user_id, $message_text);
                case 'callback':
                    return $this->handleCallbackMessage($chat_id, $user_id, $message_text);
                default:
                    return $this->handleTextMessage($chat_id, $user_id, $username, $first_name, $message_text, $user_state, $user_info);
            }
            
        } catch (Exception $e) {
            error_log("ROUTER ERROR: " . $e->getMessage());
            $this->sendErrorMessage($chat_id);
            return false;
        }
    }
    
    /**
     * 💬 مدیریت پیام‌های متنی
     */
    private function handleTextMessage($chat_id, $user_id, $username, $first_name, $message_text, $user_state, $user_info) {
        // Command routing
        if ($this->isCommand($message_text)) {
            return $this->handleCommands($chat_id, $user_id, $username, $first_name, $message_text);
        }
        
        // Button routing
        if ($this->isButton($message_text)) {
            return $this->handleButtons($chat_id, $user_id, $username, $first_name, $message_text, $user_info);
        }
        
        // State-based routing
        if ($user_state && $user_state !== 'main_menu') {
            return $this->handleStateBasedInput($chat_id, $user_id, $message_text, $user_state);
        }
        
        // Default handling
        return $this->handleDefaultInput($chat_id, $user_id, $message_text);
    }
    
    /**
     * ⚡ مدیریت کامندها
     */
    private function handleCommands($chat_id, $user_id, $username, $first_name, $command) {
        switch ($command) {
            case '/start':
                return $this->handleStartCommand($chat_id, $user_id, $username, $first_name);
                
            case '/admin':
                return $this->handleAdminCommand($chat_id, $user_id);
                
            case '/help':
                return $this->handleHelpCommand($chat_id);
                
            case '/status':
                return $this->handleStatusCommand($chat_id, $user_id);
                
            default:
                // Check for admin commands
                if ($this->isAdminCommand($command)) {
                    return $this->handleAdminSpecialCommand($chat_id, $user_id, $command);
                }
                
                return $this->handleUnknownCommand($chat_id, $command);
        }
    }
    
    /**
     * 🔘 مدیریت دکمه‌ها
     */
    private function handleButtons($chat_id, $user_id, $username, $first_name, $button_text, $user_info) {
        switch ($button_text) {
            // ═══ MAIN MENU BUTTONS ═══
            case '🎓 ورود دانشجویان':
                return $this->handleStudentLogin($chat_id, $user_id, $user_info);
                
            case '📚 شروع یادگیری':
                return $this->handleCourseRegistration($chat_id, $user_id);
                
            case '💬 پشتیبانی':
                return $this->handleSupport($chat_id);
                
            case '🌟 درباره کاپیتان':
                return $this->handleAbout($chat_id);
                
            // ═══ LOGIN BUTTONS ═══
            case '🎫 ورود با لایسنس':
                return $this->startLicenseLogin($chat_id, $user_id);
                
            case '📱 ورود با موبایل':
                return $this->startMobileLogin($chat_id, $user_id);
                
            // ═══ PAYMENT BUTTONS ═══
            case '💳 پرداخت آنلاین فوری':
                return $this->handleOnlinePayment($chat_id, $user_id);
                
            case '💰 کارت به کارت':
                return $this->handleCardPayment($chat_id, $user_id);
                
            // ═══ NAVIGATION BUTTONS ═══
            case '🔙 بازگشت':
            case '🔙 منوی اصلی':
            case '🔙 بازگشت به منوی اصلی':
            case '🏠 منوی اصلی':
                return $this->handleStartCommand($chat_id, $user_id, $username, $first_name);
                
            case '🔄 به‌روزرسانی پنل':
                return $this->refreshStudentPanel($chat_id, $user_id);
                
            // ═══ ADMIN BUTTONS ═══
            case '👨‍💼 پنل مدیریت':
                return $this->adminCommands->showMainPanel($chat_id, $user_id);
                
            case '💳 بررسی پرداخت‌ها':
                return $this->adminCommands->showPendingPayments($chat_id, $user_id);
                
            case '👥 آمار کاربران':
                return $this->adminCommands->showUserStats($chat_id, $user_id);
                
            case '📢 پیام همگانی':
                return $this->adminCommands->startBroadcast($chat_id, $user_id);
                
            case '✅ ارسال کن':
                return $this->confirmBroadcast($chat_id, $user_id);
                
            case '❌ انصراف':
                return $this->cancelBroadcast($chat_id, $user_id);
                
            // ═══ STUDENT MENU BUTTONS ═══
            default:
                return $this->handleStudentMenuButton($chat_id, $user_id, $button_text);
        }
    }
    
    /**
     * 📊 مدیریت ورودی‌های مبتنی بر State
     */
    private function handleStateBasedInput($chat_id, $user_id, $input, $state) {
        switch ($state) {
            case 'awaiting_license':
                return $this->processLicenseInput($chat_id, $user_id, $input);
                
            case 'awaiting_mobile':
                return $this->processMobileInput($chat_id, $user_id, $input);
                
            case 'awaiting_otp':
                return $this->processOTPInput($chat_id, $user_id, $input);
                
            case 'awaiting_receipt':
                return $this->processReceiptText($chat_id, $user_id, $input);
                
            case 'admin_broadcast':
                return $this->processBroadcastMessage($chat_id, $user_id, $input);
                
            case 'admin_broadcast_confirm':
                return $this->processBroadcastConfirmation($chat_id, $user_id, $input);
                
            default:
                $this->db->setUserState($user_id, 'main_menu');
                return $this->handleDefaultInput($chat_id, $user_id, $input);
        }
    }
    
    /**
     * 🚀 دستور /start
     */
    private function handleStartCommand($chat_id, $user_id, $username, $first_name) {
        $this->db->setUserState($user_id, 'main_menu');
        $this->db->registerUser($user_id, $username, $first_name, $chat_id);
        
        $message = Messages::getWelcomeMessage($first_name);
        $keyboard = Messages::getMainMenuKeyboard();
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 👨‍💼 دستور ادمین
     */
    private function handleAdminCommand($chat_id, $user_id) {
        if ($user_id != ADMIN_CHAT_ID) {
            return $this->telegram->sendMessage($chat_id, "❌ شما دسترسی ادمین ندارید.");
        }
        
        return $this->adminCommands->showMainPanel($chat_id, $user_id);
    }
    
    /**
     * 🎓 ورود دانشجویان
     */
    private function handleStudentLogin($chat_id, $user_id, $user_info) {
        if ($user_info && $user_info['is_verified']) {
            return $this->showStudentDashboard($chat_id, $user_id, $user_info['first_name']);
        }
        
        $this->db->setUserState($user_id, 'selecting_login_method');
        
        $message = Messages::getSelectLoginMethodMessage();
        $keyboard = Messages::getLoginMethodKeyboard();
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 📚 ثبت نام دوره
     */
    private function handleCourseRegistration($chat_id, $user_id) {
        if ($this->db->hasUserPurchased($user_id)) {
            return $this->telegram->sendMessage(
                $chat_id,
                "✅ شما قبلاً دوره را خریداری کرده‌اید.\n\n🎓 برای ورود: «ورود دانشجویان»",
                Messages::getMainMenuKeyboard()
            );
        }
        
        $this->db->setUserState($user_id, 'viewing_course_info');
        
        $message = Messages::getCourseInfoMessage();
        $keyboard = Messages::getPaymentMethodKeyboard();
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 💬 پشتیبانی
     */
    private function handleSupport($chat_id) {
        $message = Messages::getSupportMessage();
        $keyboard = Messages::getBackKeyboard();
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 🌟 درباره کاپیتان
     */
    private function handleAbout($chat_id) {
        $message = Messages::getAboutMessage();
        $keyboard = Messages::getBackKeyboard();
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 🎫 شروع ورود با لایسنس
     */
    private function startLicenseLogin($chat_id, $user_id) {
        $this->db->setUserState($user_id, 'awaiting_license');
        
        $message = Messages::getEnterLicenseMessage();
        $keyboard = Messages::getBackKeyboard();
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 📱 شروع ورود با موبایل
     */
    private function startMobileLogin($chat_id, $user_id) {
        $this->db->setUserState($user_id, 'awaiting_mobile');
        
        $message = Messages::getEnterMobileMessage();
        $keyboard = Messages::getBackKeyboard();
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 💳 پرداخت کارت به کارت
     */
    private function handleCardPayment($chat_id, $user_id) {
        return $this->paymentProcessor->startCardPayment($chat_id, $user_id);
    }
    
    /**
     * 🌐 پرداخت آنلاین
     */
    private function handleOnlinePayment($chat_id, $user_id) {
        $message = "🚧 <b>پرداخت آنلاین موقتاً غیرفعال</b>

━━━━━━━━━━━━━━━━━━━━

⚠️ درگاه آنلاین در حال تعمیر است

💡 <b>راه‌حل:</b>
از روش «کارت به کارت» استفاده کنید

━━━━━━━━━━━━━━━━━━━━

🔄 یا منتظر فعال‌سازی بمانید";
        
        $keyboard = Messages::getPaymentMethodKeyboard();
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 🎯 نمایش پنل دانشجویی
     */
    private function showStudentDashboard($chat_id, $user_id, $first_name = '') {
        if (empty($first_name)) {
            $user_info = $this->db->getUserInfo($user_id);
            $first_name = $user_info['first_name'] ?? 'دانشجو';
        }
        
        $menus = $this->db->getStudentMenus();
        
        $message = Messages::getStudentDashboardMessage($first_name);
        $keyboard = Messages::getStudentDashboardKeyboard($menus);
        
        return $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 📖 مدیریت دکمه‌های منوی دانشجویی
     */
    private function handleStudentMenuButton($chat_id, $user_id, $button_text) {
        $user_info = $this->db->getUserInfo($user_id);
        if (!$user_info || !$user_info['is_verified']) {
            return $this->telegram->sendMessage($chat_id, "🔐 ابتدا وارد شوید.");
        }
        
        $menus = $this->db->getStudentMenus();
        
        foreach ($menus as $menu) {
            $menu_text = $menu['icon'] . ' ' . $menu['title'];
            if ($button_text === $menu_text) {
                $this->db->logMenuAccess($user_id, $menu['id']);
                
                $keyboard = Messages::getStudentDashboardKeyboard($menus);
                return $this->telegram->sendMessage($chat_id, $menu['content'], $keyboard);
            }
        }
        
        return false;
    }
    
    /**
     * 🔍 بررسی نوع پیام
     */
    private function isCommand($text) {
        return strpos($text, '/') === 0;
    }
    
    private function isButton($text) {
        $buttons = [
            '🎓', '📚', '💬', '🌟', '🎫', '📱', '💳', '💰',
            '🔙', '🏠', '🔄', '👨‍💼', '📊', '👥', '📢', '✅', '❌'
        ];
        
        foreach ($buttons as $button) {
            if (strpos($text, $button) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    private function isAdminCommand($command) {
        $admin_commands = ['/ok', '/no', '/show', '/approve', '/reject', '/broadcast'];
        $cmd_part = explode(' ', $command)[0];
        
        return in_array($cmd_part, $admin_commands);
    }
    
    /**
     * 🎯 مدیریت کامندهای ویژه ادمین
     */
    private function handleAdminSpecialCommand($chat_id, $user_id, $command) {
        if ($user_id != ADMIN_CHAT_ID) {
            return $this->telegram->sendMessage($chat_id, "❌ دسترسی غیرمجاز");
        }
        
        $parts = explode(' ', $command);
        $cmd = $parts[0];
        $target = $parts[1] ?? '';
        
        switch ($cmd) {
            case '/ok':
            case '/approve':
                return $this->adminCommands->approvePayment($chat_id, $user_id, $target);
                
            case '/no':
            case '/reject':
                $reason = implode(' ', array_slice($parts, 2));
                return $this->adminCommands->rejectPayment($chat_id, $user_id, $target, $reason);
                
            case '/show':
                return $this->adminCommands->showPaymentDetails($chat_id, $user_id, $target);
                
            default:
                return $this->telegram->sendMessage($chat_id, "❌ کامند ادمین نامعتبر");
        }
    }
    
    /**
     * 🔤 مدیریت ورودی‌های پیش‌فرض
     */
    private function handleDefaultInput($chat_id, $user_id, $input) {
        if ($this->isGreeting($input)) {
            $response = Messages::getGreetingResponse();
            return $this->telegram->sendMessage($chat_id, $response, Messages::getMainMenuKeyboard());
        }
        
        return $this->telegram->sendMessage($chat_id, Messages::getDefaultResponse(), Messages::getMainMenuKeyboard());
    }
    
    /**
     * 👋 تشخیص سلام
     */
    private function isGreeting($text) {
        $greetings = ['سلام', 'hi', 'hello', 'درود', 'صبح بخیر', 'ظهر بخیر', 'عصر بخیر', 'شب بخیر'];
        $text_lower = mb_strtolower($text);
        
        foreach ($greetings as $greeting) {
            if (strpos($text_lower, $greeting) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * 📸 مدیریت پیام‌های عکس
     */
    private function handlePhotoMessage($chat_id, $user_id, $photo) {
        return $this->fileHandler->handlePhotoUpload($chat_id, $user_id, $photo);
    }
    
    /**
     * 📄 مدیریت پیام‌های سند
     */
    private function handleDocumentMessage($chat_id, $user_id, $document) {
        return $this->fileHandler->handleDocumentUpload($chat_id, $user_id, $document);
    }
    
    /**
     * ⚡ مدیریت کال‌بک‌ها
     */
    private function handleCallbackMessage($chat_id, $user_id, $callback_data) {
        return $this->callbackHandler->handleCallback('', $chat_id, $user_id, $callback_data);
    }
    
    /**
     * ❌ ارسال پیام خطا
     */
    private function sendErrorMessage($chat_id) {
        $message = "❌ خطایی رخ داده است.\n\n🔄 لطفاً دوباره تلاش کنید یا /start بزنید.";
        $this->telegram->sendMessage($chat_id, $message, Messages::getMainMenuKeyboard());
    }
    
    /**
     * 🎫 پردازش ورود لایسنس
     */
    private function processLicenseInput($chat_id, $user_id, $license) {
        $license = trim($license);
        
        if (strlen($license) < 50) {
            return $this->telegram->sendMessage($chat_id, Messages::getInvalidLicenseMessage(), Messages::getLoginMethodKeyboard());
        }
        
        $this->telegram->sendMessage($chat_id, "🔄 در حال بررسی لایسنس...");
        
        try {
            $spotPlayer = new SpotPlayerAPI();
            $valid = $spotPlayer->verifyLicense($license);
            
            if ($valid) {
                $this->db->setUserLicense($user_id, $license);
                $this->db->setUserState($user_id, 'authenticated');
                
                $this->telegram->sendMessage($chat_id, Messages::getLoginSuccessMessage());
                sleep(1);
                $this->showStudentDashboard($chat_id, $user_id);
            } else {
                $this->telegram->sendMessage($chat_id, Messages::getInvalidLicenseMessage(), Messages::getLoginMethodKeyboard());
                $this->db->setUserState($user_id, 'selecting_login_method');
            }
            
        } catch (Exception $e) {
            error_log("License verification error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در بررسی. دوباره تلاش کنید.", Messages::getLoginMethodKeyboard());
        }
    }
    
    /**
     * 📱 پردازش ورود موبایل
     */
    private function processMobileInput($chat_id, $user_id, $mobile) {
        $mobile = preg_replace('/[^0-9]/', '', trim($mobile));
        
        if (!preg_match('/^09\d{9}$/', $mobile)) {
            return $this->telegram->sendMessage($chat_id, Messages::getInvalidMobileMessage(), Messages::getLoginMethodKeyboard());
        }
        
        $otp = sprintf('%06d', random_int(100000, 999999));
        
        try {
            $this->db->setUserOTP($user_id, $otp, $mobile);
            $this->db->setUserState($user_id, 'awaiting_otp');
            
            $message = "📱 <b>کد تایید ارسال شد</b>

━━━━━━━━━━━━━━━━━━━━

🔐 <b>کد تایید:</b> <code>{$otp}</code>
📞 <b>موبایل:</b> <code>{$mobile}</code>

⏰ <b>اعتبار:</b> 5 دقیقه

✍️ <b>کد را در پیام بعدی وارد کنید:</b>";
            
            return $this->telegram->sendMessage($chat_id, $message, Messages::getBackKeyboard());
            
        } catch (Exception $e) {
            error_log("OTP error: " . $e->getMessage());
            return $this->telegram->sendMessage($chat_id, "❌ خطا در ارسال کد.", Messages::getLoginMethodKeyboard());
        }
    }
    
    /**
     * 🔐 پردازش کد OTP
     */
    private function processOTPInput($chat_id, $user_id, $otp) {
        $otp = trim($otp);
        
        if (!preg_match('/^\d{6}$/', $otp)) {
            return $this->telegram->sendMessage($chat_id, "❌ کد باید 6 رقم باشد.\n\n🔄 دوباره وارد کنید:");
        }
        
        try {
            if ($this->db->verifyOTP($user_id, $otp)) {
                $this->db->setUserState($user_id, 'authenticated');
                
                $this->telegram->sendMessage($chat_id, Messages::getLoginSuccessMessage());
                sleep(1);
                $this->showStudentDashboard($chat_id, $user_id);
            } else {
                $this->telegram->sendMessage($chat_id, Messages::getInvalidOTPMessage());
            }
            
        } catch (Exception $e) {
            error_log("OTP verification error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در بررسی کد.");
        }
    }
}
?>