<?php
/**
 * Complete Payment Processing System
 * Updated: 2025-10-15 19:15:26 UTC
 * Created by: saportinsta65
 */

class PaymentProcessor {
    private $telegram;
    private $db;
    private $spotPlayer;
    
    public function __construct($telegram, $db, $spotPlayer) {
        $this->telegram = $telegram;
        $this->db = $db;
        $this->spotPlayer = $spotPlayer;
    }
    
    /**
     * 🎯 شروع فرآیند پرداخت کارت به کارت
     */
    public function startCardPayment($chat_id, $user_id) {
        try {
            // Check if user already has pending payment
            $existing = $this->getPendingUserPayment($user_id);
            if ($existing) {
                $this->showExistingPayment($chat_id, $existing);
                return;
            }
            
            // Create new payment record
            $payment_id = $this->db->createPayment($user_id, COURSE_PRICE, 'card_to_card');
            
            if (!$payment_id) {
                throw new Exception("Failed to create payment record");
            }
            
            // Set user states
            $this->db->setUserState($user_id, 'awaiting_receipt');
            $this->db->setUserPaymentId($user_id, $payment_id);
            
            // Send payment instructions
            $this->sendCardPaymentInstructions($chat_id, $payment_id);
            
            // Send receipt upload guide
            $this->sendReceiptUploadGuide($chat_id);
            
            return $payment_id;
            
        } catch (Exception $e) {
            error_log("Start card payment error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در شروع پرداخت. لطفاً دوباره تلاش کنید.");
            return false;
        }
    }
    
    /**
     * 💳 ارسال دستورالعمل پرداخت کارت به کارت
     */
    private function sendCardPaymentInstructions($chat_id, $payment_id) {
        $price = number_format(COURSE_PRICE);
        
        $message = "💳 <b>پرداخت کارت به کارت</b>

━━━━━━━━━━━━━━━━━━━━

💳 <b>شماره کارت:</b>
<code>" . CARD_NUMBER . "</code>

👤 <b>نام صاحب حساب:</b>
" . CARD_HOLDER . "

💰 <b>مبلغ قابل پرداخت:</b>
🔥 <b>{$price} تومان</b>

🆔 <b>کد پیگیری شما:</b>
<code>{$payment_id}</code>

━━━━━━━━━━━━━━━━━━━━

⚡ <b>مراحل پرداخت:</b>

1️⃣ <b>کارت را کپی کنید</b>
   طولانی فشار دهید و کپی کنید

2️⃣ <b>مبلغ را واریز کنید</b>
   از اپ بانکی یا کارتخوان

3️⃣ <b>رسید را ارسال کنید</b>
   عکس رسید در همین چت

━━━━━━━━━━━━━━━━━━━━

🛡️ <b>پرداخت کاملاً امن و قابل اعتماد</b>
🏆 بیش از 10,000 پرداخت موفق

⏰ <b>بعد از واریز، رسید را بفرستید...</b>";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '📋 کپی شماره کارت',
                        'callback_data' => 'copy_card_number'
                    ]
                ],
                [
                    [
                        'text' => '🔍 وضعیت پرداخت',
                        'callback_data' => 'payment_status_' . $payment_id
                    ],
                    [
                        'text' => '❌ لغو پرداخت',
                        'callback_data' => 'payment_cancel_' . $payment_id
                    ]
                ]
            ]
        ];
        
        $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 📸 راهنمای آپلود رسید
     */
    private function sendReceiptUploadGuide($chat_id) {
        $message = "📸 <b>راهنمای ارسال رسید</b>

━━━━━━━━━━━━━━━━━━━━

✨ <b>برای تایید سریع:</b>

📷 <b>عکس‌گیری رسید:</b>
• واضح و خوانا باشد
• تمام اطلاعات دیده شود
• نور کافی داشته باشد

━━━━━━━━━━━━━━━━━━━━

📋 <b>اطلاعات مورد نیاز:</b>
✅ مبلغ واریزی
✅ تاریخ و ساعت
✅ شماره کارت مقصد
✅ نام صاحب حساب
✅ شماره پیگیری (اختیاری)

━━━━━━━━━━━━━━━━━━━━

📤 <b>نحوه ارسال:</b>
عکس را مستقیماً در همین چت بفرستید

━━━━━━━━━━━━━━━━━━━━

⏰ <b>زمان بررسی:</b> 15-30 دقیقه
🎯 <b>نکته:</b> عکس واضح = تایید سریع‌تر

🔥 <b>حالا عکس رسید خود را ارسال کنید...</b>";
        
        $this->telegram->sendMessage($chat_id, $message);
    }
    
    /**
     * 🔍 نمایش پرداخت موجود
     */
    private function showExistingPayment($chat_id, $payment) {
        $status_text = $this->getStatusText($payment['status']);
        
        $message = "💳 <b>شما پرداخت در انتظار دارید</b>

━━━━━━━━━━━━━━━━━━━━

🆔 <b>کد:</b> <code>{$payment['id']}</code>
💰 <b>مبلغ:</b> " . number_format($payment['amount']) . " تومان
🔄 <b>وضعیت:</b> {$status_text}
📅 <b>تاریخ:</b> " . date('Y/m/d H:i', strtotime($payment['created_at'])) . "

━━━━━━━━━━━━━━━━━━━━";
        
        if ($payment['status'] === 'pending') {
            $message .= "

🔄 <b>اقدامات:</b>
• اگر واریز کرده‌اید، رسید بفرستید
• یا پرداخت جدید شروع کنید";
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        [
                            'text' => '🔄 تلاش مجدد',
                            'callback_data' => 'payment_retry_' . $payment['id']
                        ],
                        [
                            'text' => '❌ لغو و شروع جدید',
                            'callback_data' => 'payment_cancel_' . $payment['id']
                        ]
                    ]
                ]
            ];
            
        } elseif ($payment['status'] === 'reviewing') {
            $message .= "

⏳ <b>رسید شما در حال بررسی است...</b>
📞 پشتیبانی: " . SUPPORT_USERNAME;
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        [
                            'text' => '🔍 بررسی وضعیت',
                            'callback_data' => 'payment_status_' . $payment['id']
                        ]
                    ]
                ]
            ];
        }
        
        $this->telegram->sendMessage($chat_id, $message, $keyboard ?? null);
    }
    
    /**
     * 🔍 دریافت پرداخت در انتظار کاربر
     */
    private function getPendingUserPayment($user_id) {
        try {
            $stmt = $this->db->connection->prepare("
                SELECT * FROM payments 
                WHERE user_id = ? 
                AND status IN ('pending', 'reviewing') 
                ORDER BY created_at DESC 
                LIMIT 1
            ");
            
            $stmt->execute([$user_id]);
            return $stmt->fetch();
            
        } catch (Exception $e) {
            error_log("Get pending payment error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🎫 ایجاد لایسنس پس از تایید پرداخت
     */
    public function createLicenseAfterPayment($payment_id, $user_id) {
        try {
            $user_info = $this->db->getUserInfo($user_id);
            if (!$user_info) {
                throw new Exception("User not found");
            }
            
            // Create license via SpotPlayer API
            $license_result = $this->spotPlayer->createLicense(
                $user_info['first_name'],
                $user_info['mobile_number'] ?? $user_info['user_id'],
                false // not test
            );
            
            if (!$license_result['success']) {
                throw new Exception("Failed to create license: " . $license_result['message']);
            }
            
            // Save license to database
            $license_data = [
                'license_id' => $license_result['license_id'],
                'license_key' => $license_result['license_key'],
                'download_url' => $license_result['download_url'] ?? 'https://spotplayer.ir',
                'course_ids' => [COURSE_ID],
                'is_test' => false
            ];
            
            $this->db->createSpotPlayerLicense($user_id, $payment_id, $license_data);
            
            // Update user as verified
            $this->db->setUserLicense($user_id, $license_result['license_key']);
            
            // Send license to user
            $this->sendLicenseToUser($user_info['chat_id'], $license_data);
            
            return $license_result;
            
        } catch (Exception $e) {
            error_log("Create license after payment error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * 📧 ارسال لایسنس به کاربر
     */
    private function sendLicenseToUser($chat_id, $license_data) {
        $message = "🎉 <b>تبریک! لایسنس شما آماده است!</b>

━━━━━━━━━━━━━━━━━━━━

✅ <b>پرداخت شما تایید شد</b>

🎫 <b>کد لایسنس:</b>
<code>{$license_data['license_key']}</code>

━━━━━━━━━━━━━━━━━━━━

📱 <b>مراحل فعال‌سازی:</b>

1️⃣ <b>دانلود اپلیکیشن:</b>
{$license_data['download_url']}

2️⃣ <b>ورود لایسنس:</b>
لایسنس را در اپلیکیشن SpotPlayer وارد کنید

3️⃣ <b>لذت ببرید:</b>
از بیش از 100 ساعت آموزش استفاده کنید

━━━━━━━━━━━━━━━━━━━━

🎯 <b>دسترسی به ربات:</b>
دستور /start بزنید و با لایسنس وارد شوید

━━━━━━━━━━━━━━━━━━━━

🎪 <b>به خانواده کاپیتان ترید خوش آمدید!</b>
💫 سفر موفقیت شما شروع شد!";
        
        $keyboard = [
            'keyboard' => [
                [
                    ['text' => '🎓 ورود به پنل دانشجویی'],
                    ['text' => '💬 پشتیبانی']
                ],
                [
                    ['text' => '🏠 منوی اصلی']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
        
        $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 📊 متن وضعیت پرداخت
     */
    private function getStatusText($status) {
        $statuses = [
            'pending' => '⏳ در انتظار رسید',
            'reviewing' => '🔍 در حال بررسی',
            'completed' => '✅ تکمیل شده',
            'rejected' => '❌ تایید نشد',
            'cancelled' => '🚫 لغو شده'
        ];
        
        return $statuses[$status] ?? '❓ نامشخص';
    }
    
    /**
     * 📊 آمار پرداخت‌ها
     */
    public function getPaymentStats() {
        try {
            $stmt = $this->db->connection->query("
                SELECT 
                    status,
                    COUNT(*) as count,
                    SUM(amount) as total_amount
                FROM payments 
                GROUP BY status
            ");
            
            return $stmt->fetchAll();
            
        } catch (Exception $e) {
            error_log("Get payment stats error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * 🧹 پاکسازی پرداخت‌های قدیمی
     */
    public function cleanupOldPayments($days = 7) {
        try {
            $stmt = $this->db->connection->prepare("
                UPDATE payments 
                SET status = 'cancelled' 
                WHERE status = 'pending' 
                AND created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            ");
            
            $stmt->execute([$days]);
            return $stmt->rowCount();
            
        } catch (Exception $e) {
            error_log("Cleanup old payments error: " . $e->getMessage());
            return 0;
        }
    }
}
?>