<?php
/**
 * مدیریت تمرین‌ها و یکپارچگی با سیستم کمپین
 * نسخه کامل و تصحیح شده - 11 اکتبر 2025
 * بدون توابع تکراری و با رفع خطاهای JSON
 */

require_once 'functions.php';
require_once 'config.php';

// بارگذاری امن فایل‌های وابسته
if (file_exists('campaign.php')) {
    require_once 'campaign.php';
}

/**
 * ✅ تابع کمکی برای پردازش ایمن JSON
 */
function safeJsonDecode($data, $default = []) {
    if (empty($data)) {
        return $default;
    }
    
    if (is_array($data)) {
        return $data;
    }
    
    if (is_string($data)) {
        $decoded = json_decode($data, true);
        return is_array($decoded) ? $decoded : $default;
    }
    
    return $default;
}

/**
 * ارسال تمرین جلسه به کاربر
 */
function sendExercise($user_id, $session_title) {
    try {
        $sessions = loadSessions();
        $session = null;
        
        foreach ($sessions as $sess) {
            if ($sess['title'] == $session_title) {
                $session = $sess;
                break;
            }
        }
        
        if (!$session || !isset($session['exercise']) || !$session['exercise']) {
            error_log("No exercise found for session: $session_title");
            return false;
        }
        
        $exercise_text = $session['exercise'];
        $message = "📝 <b>تمرین این جلسه:</b>\n\n" . $exercise_text . "\n\nلطفاً پاسخ تمرین را به صورت پیام متنی ارسال کنید.";
        
        $result = sendMessage($user_id, $message);
        
        if ($result) {
            // وضعیت تمرین را pending بگذاریم
            $user = getUserById($user_id);
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            if (!isset($exercises[$session['id']])) {
                $exercises[$session['id']] = [
                    'answer' => '',
                    'status' => 'pending',
                    'sent_at' => date('Y-m-d H:i:s')
                ];
                $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
                saveUser($user);
                
                error_log("Exercise sent to user $user_id for session: $session_title");
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Error sending exercise to user $user_id: " . $e->getMessage());
        return false;
    }
}

/**
 * پردازش پاسخ تمرین کاربر
 */
function handleExerciseAnswer($user_id, $session_title, $answer) {
    try {
        $sessions = loadSessions();
        $session = null;
        
        foreach ($sessions as $sess) {
            if ($sess['title'] == $session_title) {
                $session = $sess;
                break;
            }
        }
        
        if (!$session) {
            error_log("Session not found: $session_title");
            return false;
        }
        
        $session_id = $session['id'];
        
        // ذخیره پاسخ کاربر
        $user = getUserById($user_id);
        $exercises = safeJsonDecode($user['exercises'] ?? null, []);
        
        $exercises[$session_id] = [
            'answer' => $answer,
            'status' => 'pending',
            'submitted_at' => date('Y-m-d H:i:s')
        ];
        
        $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
        saveUser($user);
        
        // اطلاع به کاربر
        sendMessage($user_id, "✅ تمرین شما دریافت شد و در انتظار بررسی ادمین است.\n\nبه محض تایید، جلسه بعدی برای شما فعال خواهد شد.");
        
        // ارسال برای ادمین با دکمه‌های inline
        $user_info = getUserById($user_id);
        $user_name = $user_info['first_name'] ?? 'کاربر';
        
        $admin_message = "📝 <b>تمرین جدید دریافت شد</b>\n\n";
        $admin_message .= "👤 کاربر: <b>$user_name</b> (<code>$user_id</code>)\n";
        $admin_message .= "📚 جلسه: <b>$session_title</b>\n";
        $admin_message .= "📝 پاسخ:\n\n<code>$answer</code>";
        
        // ارسال پیام اول (اطلاعات)
        sendMessage(ADMIN_ID, $admin_message);
        
        // ارسال پیام دوم (دکمه‌ها)
        $reply_markup = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '✅ تایید تمرین',
                        'callback_data' => "exercise_accept_{$user_id}_{$session_id}"
                    ],
                    [
                        'text' => '❌ رد تمرین',
                        'callback_data' => "exercise_reject_{$user_id}_{$session_id}"
                    ]
                ]
            ]
        ];
        
        // ارسال دکمه‌ها با API مستقیم
        $url = API_URL . "sendMessage";
        $data = [
            'chat_id' => ADMIN_ID,
            'text' => "لطفاً تمرین را بررسی و تایید یا رد کنید:",
            'reply_markup' => json_encode($reply_markup)
        ];
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_exec($ch);
        curl_close($ch);
        
        error_log("Exercise answer submitted by user $user_id for session: $session_title");
        return true;
        
    } catch (Exception $e) {
        error_log("Error handling exercise answer for user $user_id: " . $e->getMessage());
        return false;
    }
}

/**
 * پردازش callback های تایید/رد تمرین
 */
function handleExerciseCallback($data) {
    try {
        // تایید تمرین
        if (preg_match('/^exercise_accept_([0-9]+)_([0-9]+)$/', $data, $matches)) {
            $user_id = $matches[1];
            $session_id = $matches[2];
            
            $user = getUserById($user_id);
            if (!$user) {
                error_log("User not found: $user_id");
                return false;
            }
            
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            if (isset($exercises[$session_id])) {
                $exercises[$session_id]['status'] = 'accepted';
                $exercises[$session_id]['approved_at'] = date('Y-m-d H:i:s');
                $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
                saveUser($user);
                
                // پیدا کردن نام جلسه
                $sessions = loadSessions();
                $session_title = "";
                foreach ($sessions as $sess) {
                    if ($sess['id'] == $session_id) {
                        $session_title = $sess['title'];
                        break;
                    }
                }
                
                // اطلاع به کاربر
                sendMessage($user_id, "🎉 <b>تبریک!</b>\n\nتمرین شما برای جلسه <b>$session_title</b> تایید شد!\n\nجلسه بعدی برای شما فعال شد.");
                
                // اطلاع به ادمین
                sendMessage(ADMIN_ID, "✅ تمرین کاربر <code>$user_id</code> برای جلسه <b>$session_title</b> تایید شد.");
                
                // بررسی اینکه آیا این آخرین جلسه است
                $is_last_session = isLastSession($session_id);
                
                if ($is_last_session) {
                    // بررسی اینکه آیا کاربر واجد شرایط کمپین است
                    if (function_exists('isUserEligibleForCampaign') && isUserEligibleForCampaign($user_id)) {
                        error_log("User $user_id completed the course, starting campaign");
                        
                        // شروع کمپین ایمن
                        if (function_exists('startCampaign')) {
                            startCampaign($user_id);
                            sendMessage(ADMIN_ID, "🎯 کاربر <code>$user_id</code> دوره را تکمیل کرد و کمپین برای او شروع شد.");
                        } else {
                            error_log("startCampaign function not found");
                            sendMessage(ADMIN_ID, "⚠️ کاربر <code>$user_id</code> دوره را تکمیل کرد اما تابع کمپین یافت نشد.");
                        }
                    } else {
                        error_log("User $user_id completed course but not eligible for campaign");
                        sendMessage(ADMIN_ID, "⚠️ کاربر <code>$user_id</code> دوره را تکمیل کرد اما واجد شرایط کمپین نیست.");
                    }
                }
                
                error_log("Exercise accepted for user $user_id, session $session_id");
                return true;
            } else {
                error_log("Exercise not found for user $user_id, session $session_id");
                sendMessage(ADMIN_ID, "❌ تمرین برای این کاربر و جلسه یافت نشد.");
                return false;
            }
        }
        
        // رد تمرین
        if (preg_match('/^exercise_reject_([0-9]+)_([0-9]+)$/', $data, $matches)) {
            $user_id = $matches[1];
            $session_id = $matches[2];
            
            $user = getUserById($user_id);
            if (!$user) {
                error_log("User not found: $user_id");
                return false;
            }
            
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            if (isset($exercises[$session_id])) {
                $exercises[$session_id]['status'] = 'rejected';
                $exercises[$session_id]['rejected_at'] = date('Y-m-d H:i:s');
                $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
                saveUser($user);
                
                // پیدا کردن نام جلسه
                $sessions = loadSessions();
                $session_title = "";
                foreach ($sessions as $sess) {
                    if ($sess['id'] == $session_id) {
                        $session_title = $sess['title'];
                        break;
                    }
                }
                
                // اطلاع به کاربر
                sendMessage($user_id, "❌ تمرین شما برای جلسه <b>$session_title</b> رد شد.\n\nلطفاً آموزش را مجدد مطالعه کنید و تمرین را دوباره ارسال کنید.");
                
                // اطلاع به ادمین
                sendMessage(ADMIN_ID, "❌ تمرین کاربر <code>$user_id</code> برای جلسه <b>$session_title</b> رد شد.");
                
                error_log("Exercise rejected for user $user_id, session $session_id");
                return true;
            } else {
                error_log("Exercise not found for user $user_id, session $session_id");
                sendMessage(ADMIN_ID, "❌ تمرین برای این کاربر و جلسه یافت نشد.");
                return false;
            }
        }
        
        return false;
        
    } catch (Exception $e) {
        error_log("Error handling exercise callback: " . $e->getMessage());
        return false;
    }
}

/**
 * بررسی اینکه آیا کاربر می‌تواند جلسه بعدی را ببیند
 */
function canSeeNextSession($user_id, $session_title) {
    try {
        $sessions = loadSessions();
        $current_session = null;
        
        // پیدا کردن جلسه فعلی
        foreach ($sessions as $sess) {
            if ($sess['title'] == $session_title) {
                $current_session = $sess;
                break;
            }
        }
        
        if (!$current_session) {
            return true; // اگر جلسه پیدا نشد، اجازه دسترسی بده
        }
        
        $current_id = $current_session['id'];
        
        // اگر اولین جلسه است، همیشه قابل دسترسی است
        if ($current_id == 1) {
            return true;
        }
        
        // پیدا کردن جلسه قبلی
        $previous_session = null;
        foreach ($sessions as $sess) {
            if ($sess['id'] == ($current_id - 1)) {
                $previous_session = $sess;
                break;
            }
        }
        
        if (!$previous_session) {
            return true; // اگر جلسه قبلی پیدا نشد، اجازه دسترسی بده
        }
        
        // بررسی وضعیت تمرین جلسه قبلی
        $user = getUserById($user_id);
        $exercises = safeJsonDecode($user['exercises'] ?? null, []);
        
        if (isset($exercises[$previous_session['id']]) && 
            $exercises[$previous_session['id']]['status'] == 'accepted') {
            return true;
        }
        
        return false;
        
    } catch (Exception $e) {
        error_log("Error checking session access for user $user_id: " . $e->getMessage());
        return true; // در صورت خطا، اجازه دسترسی بده
    }
}

/**
 * بررسی اینکه آیا جلسه‌ای آخرین جلسه است
 */
function isLastSession($session_id) {
    try {
        $sessions = loadSessions();
        if (empty($sessions)) {
            return false;
        }
        
        // مرتب‌سازی بر اساس ID
        usort($sessions, function($a, $b) {
            return $a['id'] - $b['id'];
        });
        
        $last_session = end($sessions);
        return $last_session['id'] == $session_id;
        
    } catch (Exception $e) {
        error_log("Error checking if session is last: " . $e->getMessage());
        return false;
    }
}

/**
 * دریافت آمار تمرین‌ها برای ادمین
 */
function getExerciseStats() {
    try {
        $users = loadUsers();
        $sessions = loadSessions();
        
        $stats = [
            'total_users' => count($users),
            'total_exercises' => 0,
            'pending_exercises' => 0,
            'accepted_exercises' => 0,
            'rejected_exercises' => 0,
            'completed_users' => 0
        ];
        
        foreach ($users as $user) {
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            $completed_count = 0;
            
            foreach ($exercises as $session_id => $exercise) {
                $stats['total_exercises']++;
                
                switch ($exercise['status'] ?? 'unknown') {
                    case 'pending':
                        $stats['pending_exercises']++;
                        break;
                    case 'accepted':
                        $stats['accepted_exercises']++;
                        $completed_count++;
                        break;
                    case 'rejected':
                        $stats['rejected_exercises']++;
                        break;
                }
            }
            
            // اگر تمام تمرین‌ها تایید شده باشند
            if ($completed_count == count($sessions)) {
                $stats['completed_users']++;
            }
        }
        
        return $stats;
        
    } catch (Exception $e) {
        error_log("Error getting exercise stats: " . $e->getMessage());
        return false;
    }
}

/**
 * ✅ دریافت لیست تمرین‌های در انتظار بررسی - نسخه تصحیح شده
 */
function getPendingExercises() {
    try {
        $users = loadUsers();
        $sessions = loadSessions();
        $pending = [];
        
        foreach ($users as $user) {
            // ✅ اصلاح: پردازش ایمن JSON برای exercises
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            if (!is_array($exercises)) {
                continue; // اگر هنوز array نیست، کاربر بعدی
            }
            
            foreach ($exercises as $session_id => $exercise) {
                if (($exercise['status'] ?? '') == 'pending') {
                    $session_title = '';
                    foreach ($sessions as $sess) {
                        if ($sess['id'] == $session_id) {
                            $session_title = $sess['title'];
                            break;
                        }
                    }
                    
                    $pending[] = [
                        'user_id' => $user['id'],
                        'user_name' => $user['first_name'] ?? 'نامشخص',
                        'session_id' => $session_id,
                        'session_title' => $session_title,
                        'answer' => $exercise['answer'] ?? '',
                        'submitted_at' => $exercise['submitted_at'] ?? 'نامشخص'
                    ];
                }
            }
        }
        
        // مرتب‌سازی بر اساس زمان ارسال (جدیدترین اول)
        usort($pending, function($a, $b) {
            $time_a = strtotime($b['submitted_at']) ?: 0;
            $time_b = strtotime($a['submitted_at']) ?: 0;
            return $time_a - $time_b;
        });
        
        return $pending;
        
    } catch (Exception $e) {
        error_log("Error getting pending exercises: " . $e->getMessage());
        return [];
    }
}

// ❌ تابع handleExerciseViewCallback() حذف شد چون در admin.php موجود است
?>